{

Fast Memory Manager: Debug Info Support DLL 1.10

Description:
 Support DLL for FastMM. With this DLL available, FastMM will report debug info
 (unit name, line numbers, etc.) for stack traces.

Usage:
 1) To compile you will need the JCL library (http://sourceforge.net/projects/jcl/)
 2) Place in the same location as the replacement borlndmm.dll or your
 application's executable module.

Change log:
 Version 1.00 (9 July 2005):
  - Initial release.
 Version 1.01 (13 July 2005):
  - Added the option to use madExcept instead of the JCL Debug library. (Thanks
    to Martin Aignesberger.)
 Version 1.02 (30 September 2005):
  - Changed options to display detail for addresses inside libraries as well.
 Version 1.03 (13 October 2005):
  - Added a raw stack trace procedure that implements raw stack traces.
 Version 1.10 (14 October 2005):
  - Improved the program logic behind the skipping of stack levels to cause
    less incorrect entries in raw stack traces. (Thanks to Craig Peterson.)

}

{--------------------Start of options block-------------------------}

{Set the option below to use madExcept instead of the JCL Debug units.}
{.$define madExcept}

{--------------------End of options block-------------------------}

library FastMM_DebugInfo;

uses
  SysUtils, Windows, {$ifndef madExcept}JCLDebug{$else}madStackTrace{$endif};

{$R *.res}

{$STACKFRAMES ON}

type
  {The state of a memory page. Used by the raw stack tracing mechanism to
   determine whether an address is a valid call site or not.}
  TMemoryPageAccess = (mpaUnknown, mpaNotExecutable, mpaExecutable);

var
  {There are a total of 1M x 4K pages in the 4GB address space}
  MemoryPageAccessMap: array[0..1024 * 1024 - 1] of TMemoryPageAccess;

{Updates the memory page}
procedure UpdateMemoryPageAccessMap(AAddress: Cardinal);
var
  LMemInfo: TMemoryBasicInformation;
  LAccess: TMemoryPageAccess;
  LStartPage, LPageCount: Cardinal;
begin
  {Query the page}
  if VirtualQuery(Pointer(AAddress), LMemInfo, SizeOf(LMemInfo)) <> 0 then
  begin
    {Get access type}
    if (LMemInfo.State = MEM_COMMIT)
      and (LMemInfo.Protect and (PAGE_EXECUTE_READ or PAGE_EXECUTE_READWRITE
        or PAGE_EXECUTE_WRITECOPY or PAGE_EXECUTE) <> 0)
      and (LMemInfo.Protect and PAGE_GUARD = 0) then
    begin
      LAccess := mpaExecutable
    end
    else
      LAccess := mpaNotExecutable;
    {Update the map}
    LStartPage := Cardinal(LMemInfo.BaseAddress) div 4096;
    LPageCount := LMemInfo.RegionSize div 4096;
    if (LStartPage + LPageCount) < Cardinal(length(MemoryPageAccessMap)) then
      FillChar(MemoryPageAccessMap[LStartPage], LPageCount, ord(LAccess));
  end
  else
  begin
    {Invalid address}
    MemoryPageAccessMap[AAddress div 4096] := mpaNotExecutable;
  end;
end;

{Returns true if the return address is a valid call site. This function is only
 safe to call while exceptions are being handled.}
function IsValidCallSite(AReturnAddress: Cardinal): boolean;
var
  LCallAddress, LCode8Back, LCode4Back: Cardinal;
begin
  if (AReturnAddress and $ffff0000 <> 0) then
  begin
    {The call address is up to 8 bytes before the return address}
    LCallAddress := AReturnAddress - 8;
    {Update the page map}
    if MemoryPageAccessMap[LCallAddress div 4096] = mpaUnknown then
      UpdateMemoryPageAccessMap(LCallAddress);
    {Check the page access}
    if (MemoryPageAccessMap[LCallAddress div 4096] = mpaExecutable)
      and (MemoryPageAccessMap[(LCallAddress + 8) div 4096] = mpaExecutable) then
    begin
      {Read the previous 8 bytes}
      try
        LCode8Back := PCardinal(LCallAddress)^;
        LCode4Back := PCardinal(LCallAddress + 4)^;
        {Is it a valid "call" instruction?}
        Result :=
          {5-byte, CALL [-$1234567]}
          ((LCode8Back and $FF000000) = $E8000000)
          {2 byte, CALL EAX}
          or ((LCode4Back and $38FF0000) = $10FF0000)
          {3 byte, CALL [EBP+0x8]}
          or ((LCode4Back and $0038FF00) = $0010FF00)
          {4 byte, CALL ??}
          or ((LCode4Back and $000038FF) = $000010FF)
          {6-byte, CALL ??}
          or ((LCode8Back and $38FF0000) = $10FF0000)
          {7-byte, CALL [ESP-0x1234567]}
          or ((LCode8Back and $0038FF00) = $0010FF00);
      except
        {The access has changed}
        UpdateMemoryPageAccessMap(LCallAddress);
        {Not executable}
        Result := False;
      end;
    end
    else
      Result := False;
  end
  else
    Result := False;
end;

{Dumps the call stack trace to the given address. Fills the list with the
 addresses where the called addresses can be found. This is the "raw" stack
 tracing routine.}
procedure GetRawStackTrace(AReturnAddresses: PCardinal; AMaxDepth, ASkipFrames: Cardinal);
var
  LStackTop, LStackBottom, LCurrentFrame, LNextFrame, LReturnAddress,
    LStackAddress: Cardinal;
begin
  {Get the call stack top and current bottom}
  asm
    mov eax, FS:[4]
    mov LStackTop, eax
    mov LStackBottom, ebp
  end;
  {Get the current frame start}
  LCurrentFrame := LStackBottom;
  {Are exceptions being handled? Can only do a raw stack trace if the possible
   access violations are going to be handled.}
  if Assigned(ExceptObjProc) then
  begin
    {Fill the call stack}
    while (AMaxDepth > 0)
      and (LCurrentFrame < LStackTop) do
    begin
      {Get the next frame}
      LNextFrame := PCardinal(LCurrentFrame)^;
      {Is it a valid stack frame address?}
      if (LNextFrame < LStackTop)
        and (LNextFrame > LCurrentFrame) then
      begin
        {The pointer to the next stack frame appears valid: Get the return
         address of the current frame}
        LReturnAddress := PCardinal(LCurrentFrame + 4)^;
        {Does this appear to be a valid return address}
        if (LReturnAddress and $ffff0000) <> 0 then
        begin
          {Is the map for this return address incorrect? If may be unknown or marked
           as unexecutable because a library was previously not yet loaded, or
           perhaps this is not a valid stack frame.}
          if MemoryPageAccessMap[(LReturnAddress - 8) div 4096] <> mpaExecutable then
            UpdateMemoryPageAccessMap(LReturnAddress - 8);
          {Is this return address actually valid?}
          if IsValidCallSite(LReturnAddress) then
          begin
            {Ignore the requested number of levels}
            if ASkipFrames = 0 then
            begin
              AReturnAddresses^ := LReturnAddress;
              Inc(AReturnAddresses);
              Dec(AMaxDepth);
            end;
          end
          else
          begin
            {If the return address is invalid it implies this stack frame is
             invalid after all.}
            LNextFrame := LStackTop;
          end;
        end
        else
        begin
          {The return address is bad - this is not a valid stack frame}
          LNextFrame := LStackTop;
        end;
      end
      else
      begin
        {This is not a valid stack frame}
        LNextFrame := LStackTop;
      end;
      {Do not check intermediate entries if there are still frames to skip}
      if ASkipFrames <> 0 then
      begin
        Dec(ASkipFrames);
      end
      else
      begin
        {Check all stack entries up to the next stack frame}
        LStackAddress := LCurrentFrame + 8;
        while (AMaxDepth > 0)
          and (LStackAddress < LNextFrame) do
        begin
          {Get the return address}
          LReturnAddress := PCardinal(LStackAddress)^;
          {Is this a valid call site?}
          if IsValidCallSite(LReturnAddress) then
          begin
            AReturnAddresses^ := LReturnAddress;
            Inc(AReturnAddresses);
            Dec(AMaxDepth);
          end;
          {Check the next stack address}
          Inc(LStackAddress, 4);
        end;
      end;
      {Do the next stack frame}
      LCurrentFrame := LNextFrame;
    end;
  end
  else
  begin
    {Exception handling is not available - a raw stack trace is not safe.}
    {Fill the call stack}
    while (AMaxDepth > 0)
      and (LCurrentFrame >= LStackBottom)
      and (LCurrentFrame < LStackTop) do
    begin
      {Ignore the requested number of levels}
      if ASkipFrames = 0 then
      begin
        AReturnAddresses^ := PCardinal(LCurrentFrame + 4)^;
        Inc(AReturnAddresses);
        Dec(AMaxDepth);
      end
      else
        Dec(ASkipFrames);
      {Get the next frame}
      LCurrentFrame := PCardinal(LCurrentFrame)^;
    end;
  end;
  {Clear the remaining dwords}
  while (AMaxDepth > 0) do
  begin
    AReturnAddresses^ := 0;
    Inc(AReturnAddresses);
    Dec(AMaxDepth);
  end;
end;

{Gets the text debug info for the given address into APDebugInfo and sets the
 number of characters returned in ANumChars. On entry ANumChars must be the
 size of the buffer pointer to by APDebugInfo.}
{$ifndef madExcept}
procedure GetDebugInfoForAddress(AAddress: Pointer;
  APDebugInfo: PChar; var ANumChars: integer);
var
  LInfo: TJCLLocationInfo;
  LTempStr: string;
begin
  GetLocationInfo(AAddress, LInfo);
  {Build the result string}
  LTempStr := ' ';
  if LInfo.SourceName <> '' then
    LTempStr := LTempStr + '[' + LInfo.SourceName + ']';
  if LInfo.UnitName <> '' then
    LTempStr := LTempStr + '[' + LInfo.UnitName + ']';
  if LInfo.ProcedureName <> '' then
    LTempStr := LTempStr + '[' + LInfo.ProcedureName + ']';
  if LInfo.LineNumber <> 0 then
    LTempStr := LTempStr + '[' + IntToStr(LInfo.LineNumber) + ']';
  {Return the result}
  if ANumChars > length(LTempStr) then
    ANumChars := length(LTempStr)
  else
    Dec(ANumChars);
  StrLCopy(APDebugInfo, PChar(LTempStr), ANumChars);
end;
{$else}
procedure GetDebugInfoForAddress(AAddress: Pointer;
  APDebugInfo: PChar; var ANumChars: integer);
var
  LTempStr: string;
begin
  LTempStr := madStackTrace.StackAddrToStr(AAddress);
  if ANumChars > length(LTempStr) then
    ANumChars := length(LTempStr)
  else
    Dec(ANumChars);
  StrLCopy(APDebugInfo, PChar(LTempStr), ANumChars);
end;
{$endif}

exports GetRawStackTrace;
exports GetDebugInfoForAddress;

begin
{$ifndef madExcept}
  JclStackTrackingOptions := JclStackTrackingOptions + [stAllModules];
{$endif}
end.
